/* Gtkdial - a GNOME frontend to wvdial
 * (c) Mike Newman/Principia Systems 1999-2001
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
 
#include <config.h>
#include <gnome.h>
#include <gdk/gdkx.h>
#include <libgnomeui/gnome-window-icon.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <fcntl.h>
#include "gtkdial.h"

char *conf;
char line[LINEBUFFER];
char default_modem[]="/var/lock/LCK..";
char select_dialer[LINEBUFFER];
void die_nice(int);
void read_line(int);
void close_mon(void);
void startup_tests(void);
void close_err(GtkWidget *, gint, gpointer);
gboolean limit_test(void);
gboolean limit_reached(void);
int wvfifo;
int pid;

static void set_service(GtkList *list,GtkWidget *widget,gpointer data);
static GtkWidget * gtk_pixmap_and_text_button(gchar *text, gchar* pixmap);

/* popt options table*/
struct poptOption options[]= {
	{"single",'s',POPT_ARG_NONE,&single,0,
	N_("Starts gtkdial in single service mode"),NULL},
	{"quiet",'q',POPT_ARG_NONE,&quiet,0,
	N_("Don't play audio alerts when (dis)connecting"),NULL},
	{"close",'c',POPT_ARG_NONE,&kill_win,0,
	N_("Close window after connecting/disconnecting"),NULL},
	{"timer",'t',POPT_ARG_NONE,&oltimer,0,
	N_("Display an online timer"),NULL},
	{"monitor",'m',POPT_ARG_NONE,&showmon,0,
	N_("Display a connection log"),NULL},
	{"exec",'e',POPT_ARG_STRING,&post_exec,0,
	N_("Execute command on successful connect"),NULL},
	{"applet",'a',POPT_ARG_NONE,&applet,0,
	N_("Work nicely with applets"),NULL},
	{NULL,'\0',0,NULL,0,NULL,NULL}
};

/* Declare the app, containers and widgets */
GList		* services=NULL;
GtkWidget	* group;
GtkWidget	* dial_bttn;
GtkWidget	* kill_bttn;
GtkWidget	* exit_bttn;
GtkWidget	* stat_line;
GtkWidget	* vbox_base;
GtkWidget	* vbox_cent;
GtkWidget	* vbox_choice;
GtkWidget	* vbox_btm;
GtkWidget	* hbox;
GtkWidget	* frame;
GtkAdjustment	* vert;
GtkAdjustment	* horiz;
GtkWidget	* scrolled;
GtkWidget	* rtclick;

GnomeUIInfo actionmenu[]={
	{GNOME_APP_UI_ITEM,N_("Dial"),NULL,
	&dial_service,NULL,0,GNOME_APP_PIXMAP_FILENAME,
	"gtkdial/dial.png",'d',GDK_CONTROL_MASK},
	{GNOME_APP_UI_ITEM,N_("Hang Up"),NULL,
	&kill_service,NULL,0,GNOME_APP_PIXMAP_FILENAME,
	"gtkdial/kill.png",'k',GDK_CONTROL_MASK},
	GNOMEUIINFO_SEPARATOR,
	{GNOME_APP_UI_ITEM,N_("Exit"),NULL,
	&check_exit,NULL,0,GNOME_APP_PIXMAP_FILENAME,
	"gtkdial/exit.png",'q',GDK_CONTROL_MASK},
	GNOMEUIINFO_END
};

GnomeUIInfo helpmenu[]={
	GNOMEUIINFO_HELP("gtkdial"),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_ABOUT_ITEM(&show_about,N_("About GtkDial")),
	GNOMEUIINFO_END
};

GnomeUIInfo timermenu[]={
	{GNOME_APP_UI_ITEM,N_("Start"),NULL,&timer_start,
	NULL,0,GNOME_APP_PIXMAP_STOCK,GNOME_STOCK_PIXMAP_TIMER,'s',
	GDK_CONTROL_MASK},
	{GNOME_APP_UI_ITEM,N_("Stop"),NULL,&timer_stop,NULL,0,
	GNOME_APP_PIXMAP_STOCK,GNOME_STOCK_PIXMAP_TIMER_STOP,'t',
	GDK_CONTROL_MASK},GNOMEUIINFO_SEPARATOR,
	{GNOME_APP_UI_ITEM,N_("Reset Session Timer"),NULL,
	&timer_reset,NULL,0,0,NULL,'r',GDK_CONTROL_MASK},
	{GNOME_APP_UI_ITEM,N_("Reset Total Timer"),NULL,
	&cumulative_reset,NULL,0,0,NULL,'z',GDK_CONTROL_MASK},
	GNOMEUIINFO_END
};

GnomeUIInfo mainpopmenu[]={
	{GNOME_APP_UI_ITEM,N_("Open Monitor"),NULL,
	&open_monitor,NULL,0},
	{GNOME_APP_UI_ITEM,N_("Close Monitor"),NULL,
	&close_mon,NULL,0},
	GNOMEUIINFO_END
};                                                                               
GnomeUIInfo prefsmenu[]={
	GNOMEUIINFO_MENU_PREFERENCES_ITEM(&settings_dlg,NULL),
        GNOMEUIINFO_END
};

GnomeUIInfo mainmenu[]={
	GNOMEUIINFO_SUBTREE(N_("_Actions"),actionmenu),
	GNOMEUIINFO_SUBTREE(N_("_Timer"),timermenu),
	GNOMEUIINFO_MENU_SETTINGS_TREE(prefsmenu),
	GNOMEUIINFO_MENU_HELP_TREE(helpmenu),
	GNOMEUIINFO_END
};


int main (int argc, char *argv[]) {	

	/* check config file for settings */	
	int conf_failed = 0;
	poptContext context;
	conf = gnome_config_get_string(SETTINGS_CONF_PATH);
	single = gnome_config_get_bool(SETTINGS_SINGLE);
	quiet = gnome_config_get_bool(SETTINGS_QUIET);
	kill_win = gnome_config_get_bool(SETTINGS_KEEP);
	oltimer = gnome_config_get_bool(SETTINGS_TIMER);
	showmon = gnome_config_get_bool(SETTINGS_MONITOR);
	reset = gnome_config_get_bool(SETTINGS_RESET);
	savetimer = gnome_config_get_bool(SETTINGS_SAVE);
	totaltimer = gnome_config_get_bool(SETTINGS_TOTAL);
	post_exec = gnome_config_get_string(SETTINGS_EXEC);
	limit = gnome_config_get_bool(SETTINGS_LIMIT);
	if (limit) {
		limit_reset = gnome_config_get_bool(SETTINGS_LIMIT_RESET);
		limit_value = 
			(gnome_config_get_int(SETTINGS_LIMIT_VALUE) * 3600);
	}
	if (savetimer) oldtimer = gnome_config_get_int(SETTINGS_CUMULATIVE);
	if (conf == NULL || strlen(conf)<2) conf=strdup("/etc/wvdial.conf");

	/* Parse the popt options, initialise the ui and
	 *  name the new toplevel */

	gnome_init_with_popt_table
	       	(PACKAGE, VERSION, argc, argv, options,0,&context);
	gnome_window_icon_set_default_from_file
		(gnome_pixmap_file("gtkdial/gnome-gtkdial.png"));
	gnome_triggers_init();
	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	textdomain(PACKAGE);
	startup_tests();	
	gtkdial = gnome_app_new (PACKAGE,"GtkDial " VERSION);
	rtclick = gnome_popup_menu_new(mainpopmenu);

	/* implement config or command line prefs */
	if (! quiet) gnome_sound_init("localhost");
	gnome_app_create_menus(GNOME_APP(gtkdial),mainmenu);

	if (oltimer || totaltimer) {
		gtk_widget_set_sensitive(GTK_WIDGET(timermenu[0].widget),1);
		gtk_widget_set_sensitive(GTK_WIDGET(timermenu[1].widget),0);
	} else {
		gtk_widget_set_sensitive(GTK_WIDGET(timermenu[0].widget),0);
		gtk_widget_set_sensitive(GTK_WIDGET(timermenu[1].widget),0);
	}

	if (oltimer) {
		gtk_widget_set_sensitive(GTK_WIDGET(timermenu[3].widget),1);
	} else {
                gtk_widget_set_sensitive(GTK_WIDGET(timermenu[3].widget),0);
	}

	if (savetimer) {
		gtk_widget_set_sensitive(GTK_WIDGET(timermenu[4].widget),1);
	} else {
		gtk_widget_set_sensitive(GTK_WIDGET(timermenu[4].widget),0);
	};

	if (showmon) {
	        gtk_widget_set_sensitive(GTK_WIDGET(mainpopmenu[0].widget),0);
	        gtk_widget_set_sensitive(GTK_WIDGET(mainpopmenu[1].widget),1);
	} else {
	        gtk_widget_set_sensitive(GTK_WIDGET(mainpopmenu[0].widget),1);
	        gtk_widget_set_sensitive(GTK_WIDGET(mainpopmenu[1].widget),0);
	};
				
	/* parse the wvdial.conf file */
	/* first, try the user prefs config_path */
	conf_failed = parse_conf(conf);

	/* If that doesn't work, try the usual place */
	if (conf_failed) {
			conf_failed=parse_conf("/etc/wvdial.conf");
	}

	/* finally, try an rc file in ~/ */
	if (conf_failed) {
			conf_failed=parse_conf
			     (g_strdup_printf("%s/.wvdialrc",getenv("HOME")));
	}

	/* we can't work with no config file, bail */
	if (conf_failed) die_nice(1);

	/* Define the main boxes */
	vbox_base = gtk_vbox_new(0,1);
	vbox_cent = gtk_vbox_new(0,1); /* for timer/total/hbox/vbox_btm  */
        timerbox  = gtk_hbox_new(0,1);
        totalbox  = gtk_hbox_new(0,1);
        hbox      = gtk_hbox_new(1,1);
        vbox_btm  = gtk_vbox_new(0,1);
	
        /* if there is more than one service to show, make a list */
        if (count>1) {
	    multi_choice();
	    gtk_container_add(GTK_CONTAINER(vbox_base),vbox_choice);
	}

	/* Add the timers */
	timer = gtk_clock_new(GTK_CLOCK_INCREASING);
	total = gtk_clock_new(GTK_CLOCK_INCREASING);
	gtk_clock_set_update_interval(GTK_CLOCK(timer),1);
        gtk_clock_set_update_interval(GTK_CLOCK(total),1);
	timer_label = gtk_label_new(_("Session Time:"));
	total_label = gtk_label_new(_("Total Time:"));
	gtk_box_pack_start(GTK_BOX(timerbox),timer_label,1,1,0);
	gtk_box_pack_end(GTK_BOX(timerbox),timer,1,1,0);
        gtk_box_pack_start(GTK_BOX(totalbox),total_label,1,1,0);
	gtk_box_pack_end(GTK_BOX(totalbox),total,1,1,0);
	if (oltimer) gtk_container_add(GTK_CONTAINER(vbox_cent),timerbox);
	if (totaltimer) gtk_container_add(GTK_CONTAINER(vbox_cent),totalbox);

	/* Add the buttons */
        dial_bttn = gtk_pixmap_and_text_button(_("Dial"),"gtkdial/dial.png");
        kill_bttn = gtk_pixmap_and_text_button(_("Hang Up"),"gtkdial/kill.png");        
	exit_bttn = gtk_pixmap_and_text_button(_("Exit"),"gtkdial/exit.png");
	gtk_box_pack_start (GTK_BOX(hbox),dial_bttn,1,1,0);
	gtk_box_pack_start (GTK_BOX(hbox),kill_bttn,1,1,0);
	gtk_box_pack_start (GTK_BOX(hbox),exit_bttn,1,1,0);
        gtk_container_add (GTK_CONTAINER(vbox_cent),hbox);

	/* Add the status bar */
        stat_line = gtk_statusbar_new();
	gtk_box_pack_end (GTK_BOX(vbox_btm),stat_line,0,0,0);
	gtk_container_add (GTK_CONTAINER(vbox_cent),vbox_btm);

	/* Insert this section into base box */
	gtk_box_pack_start(GTK_BOX(vbox_base),vbox_cent,0,0,0); 

        /* Decide what the status bar should initially say */
        if (connected()) statmsg_push(_("Connected"));
        else statmsg_push(_("Ready"));

        /* Set the initial cumulative time if required */
        if (savetimer) gtk_clock_set_seconds(GTK_CLOCK(total),oldtimer);
        if (connected()) timer_start();

        /* Connect signals to the window & interactive widgets */
        gtk_signal_connect(GTK_OBJECT(exit_bttn),"clicked",
           GTK_SIGNAL_FUNC(check_exit),NULL);

        gtk_signal_connect(GTK_OBJECT(kill_bttn),"clicked",
           GTK_SIGNAL_FUNC(kill_service),NULL);

        gtk_signal_connect(GTK_OBJECT(dial_bttn),"clicked",
           GTK_SIGNAL_FUNC(dial_service),NULL);

        gtk_signal_connect(GTK_OBJECT(gtkdial),"delete_event",
           GTK_SIGNAL_FUNC(delete_dummy),NULL);

        gtk_signal_connect(GTK_OBJECT(gtkdial),"destroy",
           GTK_SIGNAL_FUNC(check_exit),NULL);

	if (showmon) open_monitor();
	bttn_swap(connected());
	gnome_app_set_contents (GNOME_APP(gtkdial),vbox_base);
	gnome_popup_menu_attach(rtclick,gtkdial,NULL);
	gtk_window_set_policy(GTK_WINDOW(gtkdial),0,1,0);
	gtk_widget_show_all (gtkdial);

	/* if we have a limit, start a timer to check for it */
	
	gtk_main();
	exit (0);
}

/* This grabs the selected option in multi service mode */
void set_service(GtkList *list,GtkWidget *widget,gpointer data) {
	gchar *selected;
	gtk_label_get(	(GTK_LABEL(GTK_BIN(widget)->child)),
			&selected);
	strcpy(select_dialer,selected);
}
	
/* This sets up a scrolled list widget for multi service mode */
void multi_choice(void) {
    vbox_choice=gtk_vbox_new(0,1);
    horiz=GTK_ADJUSTMENT(gtk_adjustment_new(0,0,0,0,0,0));
    vert=GTK_ADJUSTMENT(gtk_adjustment_new(0,0,0,0,0,0));
    frame=gtk_frame_new(_("Select a service:"));
    gtk_box_pack_start(GTK_BOX(vbox_choice),frame,1,1,5);
    group=gtk_list_new();
    services=g_list_first(services);
    gtk_list_append_items(GTK_LIST(group),services);
    scrolled=gtk_scrolled_window_new(horiz,vert);
    gtk_scrolled_window_set_policy
	 (GTK_SCROLLED_WINDOW(scrolled),GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled),group);
    gtk_container_add(GTK_CONTAINER(frame),scrolled);
    gtk_signal_connect(	GTK_OBJECT(group),"select_child",
		 	GTK_SIGNAL_FUNC(set_service),NULL);
    gtk_list_select_item(GTK_LIST(group),0);
}	

/* A sub to make a system call to wvdial with our choice of service
 * Note that this sub will loop until pppd comes up, but will die if the
 * modem lock file is removed, so we don't hang forever for connections
 * which have been killed by the user or system */
int dial_service(void) {
	char cmdline[LINEBUFFER];
	int wvpipe;
	gchar *status_msg;
	if (connected()) return(0);
	bttn_swap(1);
	if(wvfifo) close(wvfifo);
	status_msg = g_strdup_printf(_("Connecting to %s ..."),select_dialer);
	statmsg_push(status_msg);
	g_free(status_msg);
	wvpipe=mkfifo("/tmp/gtkdial.fifo",0777);
	if (wvpipe != 0) die_nice(4);
	sprintf(cmdline,"wvdial %s > /tmp/gtkdial.fifo 2>&1 &",select_dialer);
	wvfifo=open("/tmp/gtkdial.fifo", O_RDONLY | O_NONBLOCK);
	if (wvfifo<0) showmon=0;
	system(cmdline);
	
	/* wait while shell/wvdial is spawned */
	do {	
		usleep(10000);
		read_line(wvfifo);
		gtk_main_iteration_do(0);
		gtk_events_pending();
		} while (connected()==0);
	
	/* loop until PPP comes up */
	do {
		usleep(10000);	
		read_line(wvfifo);
		gtk_main_iteration_do(0);
		gtk_events_pending();
		/* ditch if user/system killed connection*/
		if (connected()==0) break;
	} while (wait_pid()==FALSE);
	
	if (connected()==0) return (1);
	status_msg = g_strdup_printf(_("Connected to %s"),select_dialer);
	statmsg_push(status_msg);
	g_free(status_msg);
	if (reset) timer_reset();
	if (oltimer) timer_start();
	play_alert ("connect");
	read_line (wvfifo);
	gtk_main_iteration_do(0);
	gtk_events_pending();
	if (post_exec != NULL) {
		system(g_strdup_printf("%s %s",post_exec," &"));
	}
	if (kill_win) exit(0);
	if (applet) XIconifyWindow(gdk_display, GDK_WINDOW_XWINDOW (gtkdial->window),
			DefaultScreen(gdk_display));
	return (0);
}

/* A sub to kill an existing PPP link by terminating its calling wvdial
 * process as returned by connected() */
	int kill_service(void) {
	pid=connected();
	if (! pid) return(1);
	bttn_swap(0);
	kill(pid,15);
	if(kill_win) exit(0);
	statmsg_push(_("Disconnected"));
	if (oltimer) timer_stop();
	play_alert("kill");
	if (wvfifo>0) {
		read_line(wvfifo);
		close(wvfifo);
	};
	unlink ("/tmp/gtkdial.fifo");
	 if (applet) {
		 gdk_window_raise(GTK_WIDGET(gtkdial)->window);
		 gdk_flush();
		 gdk_window_show(GTK_WIDGET(gtkdial)->window);
		 gdk_flush();
	 }
	return (0);
}

/* read a line from a pipe and do stuff with it */
void read_line(int file) {
	char printline[LINEBUFFER];
	memset(printline,'\0',sizeof(printline));
	if (read(file,printline,sizeof(printline-1))>0) {
	
		/* if the monitor window is open, show the line */
		if (showmon) gtk_text_insert
				(GTK_TEXT(less),NULL,NULL,NULL,printline,-1);
	
	};
	return;
};

/* Parse the existing wvdial.conf file for service names and
 * default modem settings */

int parse_conf(gchar *config_file) {
	
	/* first, get the Modem setting from the Default Dialer section */
	/* we need this to check for a modem lock file in /var/lock/ */
	gchar temp[LINEBUFFER];
	gchar titlestr[LINEBUFFER];
	gchar list[LINEBUFFER];
	gboolean default_can_dial = FALSE;
	GtkListItem *newitem;
	FILE *fp = fopen(config_file,"r");
	if (fp==NULL) return 1;
	while(fgets(line,LINEBUFFER,fp)){
		if(sscanf(line,"[Dialer Defaults%s",temp)) {
		 while(fgets(line,LINEBUFFER,fp)) {
		 if (sscanf(line,"Phone = %s", temp))
			 		default_can_dial=TRUE;
		 if (sscanf(line,"Modem = /dev/%s",temp)) {
			gotmod = 1;
			strcat(default_modem,temp);
		 	}
		 if (line[0]=='\n') break;
		}
	    }
	}

	rewind(fp);
	
	/* Now scan the file for [Dialer ISP] sections and keep them */
	while(fgets(line,LINEBUFFER,fp)){
	 if(sscanf(line,"[Dialer %s]",list)) {
 	  if (strstr(list,"Defaults") && ! default_can_dial) continue;
	   else {
		for(j=0;j<strlen(list);j++) {if (list[j]==']') list[j]='\0';}
		newitem=(GTK_LIST_ITEM(gtk_list_item_new_with_label(list)));
		services = g_list_append(services,newitem);
		count++;
	
	 /* if user has selected -s, limit the list to one ISP */
		 if (count==1 && single) break;
		}
	   }
	  }

	/* Check for unconfigured or broken wvdial config file */
	if (!count) die_nice(2);
	if (!gotmod) die_nice(3);

	/* single user mode dialer and window title settings */
	if (count==1) {
		strcpy(select_dialer,list);
		sprintf (titlestr,"Gtkdial - %s",list);
		gtk_window_set_title(GTK_WINDOW(gtkdial),titlestr);	
	}

	return (0);
}


/* get_state() takes a pointer to a toggle button and returns a
 * gboolean reflecting its current state */
gboolean get_state(GtkWidget *widget) {
	gboolean state = 
		gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)); 
	return state;
}

/* Connection Monitor Window */
void open_monitor(void) {
	less= gtk_text_new(NULL,NULL);
	lessbox = gtk_vbox_new(0,1);
	gtk_container_add(GTK_CONTAINER(lessbox),less);
	gtk_widget_show_all(lessbox);
	gtk_container_add(GTK_CONTAINER(vbox_base),lessbox);
	showmon=1;
	gtk_widget_set_sensitive(GTK_WIDGET(mainpopmenu[0].widget),0);
	gtk_widget_set_sensitive(GTK_WIDGET(mainpopmenu[1].widget),1);
}
	
/* Close connection monitor window */
void close_mon(void) {
	gtk_widget_destroy(GTK_WIDGET(less));
	gtk_widget_destroy(GTK_WIDGET(lessbox));
	showmon=0;
	gtk_main_iteration_do(0);
	gtk_widget_set_sensitive(GTK_WIDGET(mainpopmenu[0].widget),1);
	gtk_widget_set_sensitive(GTK_WIDGET(mainpopmenu[1].widget),0);
}

void show_about(void) {
	GtkWidget *about;
	GdkImlibImage *image;
	GtkWidget *pixmap;
	GtkWidget *vbox;
	about = gnome_dialog_new(_("About Gtkdial"),
				 GNOME_STOCK_BUTTON_CLOSE,
				 NULL);
	image = gdk_imlib_load_image
			(gnome_pixmap_file("gtkdial/gtkdial_help.png"));
	pixmap=gnome_pixmap_new_from_imlib(image);
	vbox=GNOME_DIALOG(about)->vbox;
	gtk_box_pack_start(GTK_BOX(vbox),pixmap,0,0,0);
	gtk_signal_connect(GTK_OBJECT(about),"clicked",
			   GTK_SIGNAL_FUNC(gtk_main_quit),NULL);
	gtk_window_set_modal(GTK_WINDOW(about),1);
	gtk_widget_show_all(about);
	gtk_main();
	gtk_widget_destroy(about);
	return;
}

/* Find out if a modem lock file exists and return the PID of the
 * wvdial process which started things off */

gint connected(void) {
	int retval;
	char temp[LINEBUFFER];
	FILE *wvpid=fopen(default_modem,"r");
	/* if theres no lock file, we're not connected so return ZERO */
	if (wvpid==NULL) return(0);
	fgets(temp,LINEBUFFER,wvpid);
	/* get the pid from the modem lock file */
	retval=strtol(temp,NULL,0);
	fclose(wvpid);
	return retval;   
}

/* Using connected(), decide to make Dial and Kill sensitive or otherwise
 * and gray out irrelevant menubar options */
void bttn_swap(gboolean is_active) {
	if (is_active) {
		gtk_widget_set_sensitive(dial_bttn,0);
		gtk_widget_set_sensitive(kill_bttn,1);
		gtk_widget_set_sensitive(GTK_WIDGET(actionmenu[0].widget),0);
		gtk_widget_set_sensitive(GTK_WIDGET(actionmenu[1].widget),1);
	} else {
		gtk_widget_set_sensitive(dial_bttn,1);
		gtk_widget_set_sensitive(kill_bttn,0);
		gtk_widget_set_sensitive(GTK_WIDGET(actionmenu[0].widget),1);
		gtk_widget_set_sensitive(GTK_WIDGET(actionmenu[1].widget),0);
	}
	gtk_events_pending();
}

/* checks for a PPP pid file in /var/run, which in conjunction with connected()
 * tells us that the modem is engaged AND a link is up */
gboolean wait_pid(void) {
	FILE *ppp=fopen("/var/run/ppp0.pid","r");
	/* again an error on fopen means not yet, so return FALSE */
	if (ppp==NULL) return FALSE;
	fclose(ppp);
	return TRUE;
}

/* Handle status bar messages */
void statmsg_push(char *message_txt) {
	int con_id = gtk_statusbar_get_context_id
		(GTK_STATUSBAR(stat_line),"user");
	gtk_statusbar_pop(GTK_STATUSBAR(stat_line),con_id);
	gtk_statusbar_push(GTK_STATUSBAR(stat_line),con_id,message_txt);
	gtk_events_pending();
}

/* Popup "are you sure" dialog when closing during an active connection */
void check_exit(void) {
	if (connected()==0) implement_check_choice(GNOME_YES,NULL);
	gnome_question_dialog_modal_parented 
		(_("You're Connected - Really close?"),implement_check_choice,
		 NULL,GTK_WINDOW(gtkdial));
}

/* signal handler for "really close?" dialog */
void implement_check_choice(gint state, gpointer data) {
	if (state==GNOME_NO) return;
	if (oltimer) timer_stop();
	close(wvfifo);
	unlink("/tmp/gtkdial.fifo");
	unlink("/tmp/gtkdial.lock");
	gtk_main_quit();
	exit(0);
}

/* fiddle with esd a bit */
void play_alert(const gchar *filename) {
	if (quiet) return;
	gnome_triggers_do(NULL,NULL,"gtkdial",filename,NULL);
}

/* dummy call back for WM delete event */
gboolean delete_dummy(GtkWidget *widget,GdkEvent *event,gpointer data) {
	check_exit();
	return TRUE;
}

/* close callback for error dialog only */
void close_err(GtkWidget *dialog,gint button,gpointer data){
	gtk_main_quit();
};

/* trap some common errors, with a default exit case */
void die_nice(int error) {
	GtkWidget *errordialog;
	gchar *error_str;
	switch (error) {
	case 1 : {error_str=g_strdup(_("No wvdial config file was available in\n"				   				     "any of the usual places.\n\n"
				       "Run gwvedit to set one up."));
		 break;}
	case 2 : {error_str=g_strdup(_("The config file has no valid services.\n"				    				      "Run gwvedit to set them up"));
		 break;}
	case 3 : {error_str=g_strdup(
			 _("The config file had no information on your "
			   "modem.\nRun gwvedit to autodetect settings."));
		 break;}
	case 4 : {error_str=g_strdup(_("Error: wvdial error\n"));
		 break;}
	case 5 : {error_str=g_strdup(_("A Gtkdial is already running\n"));
		 break;}
	default : error_str=g_strdup(
				  _("Error: sanity check - what's going on?\n"));
  	};
	errordialog = gnome_message_box_new
			(error_str,GNOME_MESSAGE_BOX_ERROR,
			 GNOME_STOCK_BUTTON_CLOSE,NULL);
	gtk_signal_connect(GTK_OBJECT(errordialog),"clicked",close_err,NULL);
	gtk_widget_show_all(errordialog);
	gtk_main();
	g_free(error_str);
	exit (error);
}

/* timer functions for gtkdial */ 

/* start the gtkclock */
void timer_start(void) {
	gtk_clock_start(GTK_CLOCK(timer));
	gtk_clock_start(GTK_CLOCK(total));
	gtk_widget_set_sensitive(GTK_WIDGET(timermenu[0].widget),0);
	gtk_widget_set_sensitive(GTK_WIDGET(timermenu[1].widget),1);
	 if (limit) {
	             limit_check=gtk_timeout_add
			     (20000,(GtkFunction)limit_test,NULL);
	}
	 
}

/* stop the timer. Once stopped, its value is a time_t in 
 * widget.stopped, so we save it if we need to */
void timer_stop(void) {
	gtk_clock_stop(GTK_CLOCK(timer));
	gtk_clock_stop(GTK_CLOCK(total));
	gtk_widget_set_sensitive(GTK_WIDGET(timermenu[0].widget),1);
	gtk_widget_set_sensitive(GTK_WIDGET(timermenu[1].widget),0);
	if (savetimer) {gnome_config_set_int
			("gtkdial/timer/cumulative",GTK_CLOCK(total)->stopped);
			gnome_config_sync();
	}
	if (limit && limit_check) {
		gtk_timeout_remove(limit_check);
	}
}

/* zero the displayed timer */
void timer_reset(void) {
	gtk_clock_set_seconds(GTK_CLOCK(timer),0);
}

/* zero the remembered timer value */
void cumulative_reset(void) {
	gnome_config_set_int("gtkdial/timer/cumulative",0);
	oldtimer=0;
	gtk_clock_set_seconds(GTK_CLOCK(total),0);
	gnome_config_sync();
}

/* check for running gtkdial on startup and
 * remove stale fifos */

void startup_tests(void) {
	FILE *lock;
	FILE *proc;
	gchar *pid_lock;
	pid_t me=getpid();
	

	/* first, is there already a gtkdial running? */	
	/* start by checking for a lockfile */
	lock=fopen("/tmp/gtkdial.lock","r");
	if (lock!=NULL) {
		char pidline[21];
		char *procfile = NULL;
		fgets(pidline,20,lock);
		procfile=g_strdup_printf("/proc/%s/status",pidline);
		
		/* check pid still exists */
		proc = fopen(procfile,"r");
		if (proc!=NULL) {	
			/* it does - don't start up */	
			fclose(lock);
			fclose(proc);
			die_nice(5);
		}
		/* ok, its a zombie - delete the lock file */
		unlink("/tmp/gtkdial.lock");
	}
	
	/* any stale fifo's to clean up */
	unlink("/tmp/gtkdial.fifo");
	
	/* ok, we're going for it, so lock gtkdial
	* the tmp file contains our pid */
	lock=fopen("/tmp/gtkdial.lock","w");
	if (lock==NULL) {
		die_nice(6);
	}
	pid_lock = g_strdup_printf("%d",me);
	fputs(pid_lock,lock);
	fclose(lock);
	g_free(pid_lock);
}

/* this sucks so badly - periodically check that total timer does not
 * equal or exceed the limit, and warn with a dialog *
 * 
 * nasty messing with GtkClock internals to do this */

gboolean limit_test(void) {
	gboolean retval = TRUE;
	long int elapsed = ((long int)time(NULL)-GTK_CLOCK(total)->seconds);
	if (elapsed >= limit_value) {
		retval = limit_reached();
	}
	if (showmon) read_line(wvfifo);
	return retval;
}

gboolean limit_reached(void) {
	GtkWidget *dialog;
	gchar *msg;
	gboolean retval;
	play_alert("alarm");
	if (limit_reset) {
		msg = g_strdup_printf	
		  (_("Your total online time has reached %d hour(s)\n"
			 "and will now be reset"),
		gnome_config_get_int(SETTINGS_LIMIT_VALUE));
		gtk_clock_set_seconds(GTK_CLOCK(total),0);
		retval=TRUE;
	} else {
                msg = g_strdup_printf
                  (_("Your total online time has reached %d hour(s)\n"),
                  gnome_config_get_int(SETTINGS_LIMIT_VALUE));
 	        retval=FALSE;
        }			
	dialog=gnome_message_box_new(msg,GNOME_MESSAGE_BOX_WARNING,
			GNOME_STOCK_BUTTON_CLOSE,NULL);
	gtk_widget_show_all(dialog);
	g_free(msg);
	return (retval);
}

/* utility function to pack a pixmap and a label into a button */

GtkWidget * gtk_pixmap_and_text_button (gchar *text, gchar *pixmap_filename) {
	
        GtkWidget *button;
        GtkWidget *hbox;
        GtkWidget *pixmap;
        GtkWidget *label;
	gchar * path_to_pixmap = gnome_pixmap_file(pixmap_filename);
        button = gtk_button_new();
        hbox = gtk_hbox_new(0,3);
        pixmap = gnome_pixmap_new_from_file(path_to_pixmap);
        label = gtk_label_new(text);
        gtk_box_pack_start(GTK_BOX(hbox),pixmap,1,1,0);
        gtk_box_pack_end(GTK_BOX(hbox),label,1,1,0);
        gtk_container_add(GTK_CONTAINER(button),hbox);
	g_free(path_to_pixmap);
        return button;
}

